# @package      hubzero-submit-distributor
# @file         EnvironmentWhitelistInfo.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os.path
import re
import glob
import json
import logging

from hubzero.submit.LogMessage      import getLogJobIdMessage as getLogMessage
from hubzero.submit.GroupMembership import GroupMembership

try:
   iterRange = xrange
except NameError as e:
   iterRange = range

class EnvironmentWhitelistInfo:
   def __init__(self,
                whiteListPath,
                restrictionUser=None):
      self.logger = logging.getLogger(__name__)

      self.groupsEnvironmentWhiteList = {}
      self.environmentWhiteList       = []

      if os.path.isdir(whiteListPath):
         for whiteListInfoPath in glob.iglob(os.path.join(whiteListPath,'*')):
            self.readEnvironmentWhitelistInfoFile(whiteListInfoPath)
      else:
         for whiteListInfoPath in glob.iglob(whiteListPath):
            self.readEnvironmentWhitelistInfoFile(whiteListInfoPath)

      groupMembership = GroupMembership(restrictionUser=restrictionUser)
      markedForDeletion = []
      for groupName in self.groupsEnvironmentWhiteList:
         if not groupMembership.isGroupMember(groupName):
            markedForDeletion.append(groupName)
      for groupName in markedForDeletion:
         del self.groupsEnvironmentWhiteList[groupName]
      del markedForDeletion

      for groupName in self.groupsEnvironmentWhiteList:
         markedForDeletion = []
         for groupIndex in iterRange(len(self.groupsEnvironmentWhiteList[groupName])):
            if self.groupsEnvironmentWhiteList[groupName][groupIndex]['state'] != 'enabled':
               markedForDeletion.append(groupIndex)
         markedForDeletion.reverse()
         for groupIndex in markedForDeletion:
            del self.groupsEnvironmentWhiteList[groupName][groupIndex]
         del markedForDeletion

      for groupName in self.groupsEnvironmentWhiteList:
         for groupIndex in iterRange(len(self.groupsEnvironmentWhiteList[groupName])):
            for environmentVariable in self.groupsEnvironmentWhiteList[groupName][groupIndex]['environmentVariables']:
               if not environmentVariable in self.environmentWhiteList:
                  self.environmentWhiteList.append(environmentVariable)


   def readEnvironmentWhitelistInfoFile(self,
                                        environmentWhitelistInfoPath):
      groupPattern    = re.compile(r'(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile(r'( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile(r'\s*#.*')
      groupName       = ""

      if os.path.exists(environmentWhitelistInfoPath):
         try:
            fpInfo = open(environmentWhitelistInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   groupPattern.match(record):
                        groupName = groupPattern.match(record).group(2)
                        if not groupName in self.groupsEnvironmentWhiteList:
                           self.groupsEnvironmentWhiteList[groupName] = []
                        groupIndex = len(self.groupsEnvironmentWhiteList[groupName])
                        self.groupsEnvironmentWhiteList[groupName].append({'environmentVariables':[],
                                                                           'state':'enabled'
                                                                          })
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.groupsEnvironmentWhiteList[groupName][groupIndex]:
                           if   isinstance(self.groupsEnvironmentWhiteList[groupName][groupIndex][key],list):
                              self.groupsEnvironmentWhiteList[groupName][groupIndex][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.groupsEnvironmentWhiteList[groupName][groupIndex][key],bool):
                              self.groupsEnvironmentWhiteList[groupName][groupIndex][key] = bool(value.lower() == 'true')
                           elif isinstance(self.groupsEnvironmentWhiteList[groupName][groupIndex][key],float):
                              self.groupsEnvironmentWhiteList[groupName][groupIndex][key] = float(value)
                           elif isinstance(self.groupsEnvironmentWhiteList[groupName][groupIndex][key],int):
                              self.groupsEnvironmentWhiteList[groupName][groupIndex][key] = int(value)
                           elif isinstance(self.groupsEnvironmentWhiteList[groupName][groupIndex][key],dict):
                              try:
                                 sampleKey   = list(self.groupsEnvironmentWhiteList[groupName][groupIndex][key].keys())[0]
                                 sampleValue = self.groupsEnvironmentWhiteList[groupName][groupIndex][key][sampleKey]
                              except:
                                 try:
                                    self.groupsEnvironmentWhiteList[groupName][groupIndex][key] = json.loads(value)
                                 except:
                                    self.groupsEnvironmentWhiteList[groupName][groupIndex][key] = {}
                                    sampleKey   = "key"
                                    sampleValue = "value"
                              else:
                                 self.groupsEnvironmentWhiteList[groupName][groupIndex][key] = {}
               
                              if not self.groupsEnvironmentWhiteList[groupName][groupIndex][key]:
                                 for e in value.split(','):
                                    dictKey,dictValue = e.split(':')
                                    if isinstance(sampleKey,int):
                                       dictKey = int(dictKey)
                                    if   isinstance(sampleValue,int):
                                       dictValue = int(dictValue)
                                    elif isinstance(sampleValue,float):
                                       dictValue = float(dictValue)
                                    elif isinstance(sampleValue,bool):
                                       dictValue = bool(dictValue.lower() == 'true')
                                    self.groupsEnvironmentWhiteList[groupName][groupIndex][key][dictKey] = dictValue
                           else:
                              self.groupsEnvironmentWhiteList[groupName][groupIndex][key] = value
                        else:
                           self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s for group %s" % \
                                                                                                     (key,value,groupName)))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Environment white list file %s could not be read" % \
                                                                                 (environmentWhitelistInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Environment white list file %s could not be opened" % \
                                                                                (environmentWhitelistInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Environment white list file %s is missing" % \
                                                                    (environmentWhitelistInfoPath)))


   def isVariableInWhiteList(self,
                             variable):
      return(variable in self.environmentWhiteList)


