#!/usr/bin/env python
#
# @package      hubzero-submit-monitors
# @file         Scripts/boinc/uploadFiles.py
# @copyright    Copyright (c) 2017-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2017-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
# ----------------------------------------------------------------------
#  uploadFiles.py
#
#  script which uploads job input files
#
import sys
import os

from submit_api import *

BOINCPROJECTURL    = ""
BOINCAUTHENTICATOR = ""


class UploadBoincFiles:
   def __init__(self,
                boincProjectURL,
                boincAuthenticator,
                runName,
                localJobId,
                batchId):
      self.boincProjectURL    = boincProjectURL
      self.boincAuthenticator = boincAuthenticator
      self.runName            = runName
      self.localJobId         = localJobId
      self.batchId            = int(batchId)


   def run(self):
      exitStatus = 0
      fileUploadRequest = UPLOAD_FILES_REQ()
      fileUploadRequest.project       = self.boincProjectURL
      fileUploadRequest.authenticator = self.boincAuthenticator
      fileUploadRequest.batch_id      = self.batchId

      jobDirectory = os.getcwd()
      if os.path.basename(jobDirectory) != self.localJobId and \
         os.path.basename(jobDirectory) != self.runName:
         jobDirectory = os.path.dirname(jobDirectory)
      if os.path.basename(jobDirectory) != self.localJobId and \
         os.path.basename(jobDirectory) != self.runName:
         exitStatus = 1
         sys.stderr.write("uploadFiles.py must be run in the job directory: %s\n" % (os.getcwd()))
      else:
         workDirectory = os.path.dirname(jobDirectory)
         instanceIds = []
         dirFiles = os.listdir(jobDirectory)
         for dirFile in dirFiles:
            dirPath = os.path.join(jobDirectory,dirFile)
            if os.path.isdir(dirPath):
               try:
                  instanceNumber = int(dirFile)
               except:
                  pass
               else:
                  instanceIds.append(dirFile)
         instanceIds.sort()

         fileUploadRequest.local_names = ()
         fileUploadRequest.boinc_names = ()
         for instanceId in instanceIds:
            inputTarFile =  '_'.join([self.localJobId,instanceId,'input.tar.gz'])
            stagedTarFilePath = os.path.join(jobDirectory,instanceId,inputTarFile)
            if os.path.exists(stagedTarFilePath):
               fileUploadRequest.local_names += (stagedTarFilePath,)
               fileUploadRequest.boinc_names += (inputTarFile,)
            else:
               exitStatus = 1
               errMessage = "%s is missing" % (stagedTarFilePath)
               sys.stderr.write("%s\n" % (errMessage))

         if not exitStatus:
            fileUploadResponse = upload_files(fileUploadRequest)

            try:
               if fileUploadResponse.find('error') is not None:
                  exitStatus = 1
                  errMessage = fileUploadResponse.find('error').find('error_msg').text
                  sys.stderr.write("%s\n" % (errMessage))
            except:
               exitStatus = 1
               errMessage = fileUploadResponse.text
               sys.stderr.write("%s\n" % (errMessage))

      return(exitStatus)


if __name__ == '__main__':

   if len(sys.argv) == 4:
      runName    = sys.argv[1]
      localJobId = sys.argv[2]
      batchId    = sys.argv[3]
   else:
      sys.exit(1)

   __uploadBoincFiles__ = UploadBoincFiles(BOINCPROJECTURL,BOINCAUTHENTICATOR,
                                           runName,localJobId,batchId)
   __exitStatus__ = __uploadBoincFiles__.run()

   sys.exit(__exitStatus__)


