# @package      hubzero-submit-distributor
# @file         Tapis3SitesInfo.py
# @copyright    Copyright (c) 2020-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2020-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import re
import glob
import json
import logging

from hubzero.submit.LogMessage      import getLogJobIdMessage as getLogMessage
from hubzero.submit.GroupMembership import GroupMembership

class Tapis3SitesInfo:
   def __init__(self,
                tapisSitesPath,
                restrictionUser=None):
      self.logger = logging.getLogger(__name__)

      self.tapisSites = {}

      if os.path.isdir(tapisSitesPath):
         for tapisSitesInfoPath in glob.iglob(os.path.join(tapisSitesPath,'*')):
            self.readTapis3SitesInfoFile(tapisSitesInfoPath)
      else:
         for tapisSitesInfoPath in glob.iglob(tapisSitesPath):
            self.readTapis3SitesInfoFile(tapisSitesInfoPath)

      if restrictionUser:
         for tapisSite in self.tapisSites:
            restrictedToUsers = self.tapisSites[tapisSite]['restrictedToUsers']
            if len(restrictedToUsers) > 0:
               if not restrictionUser in restrictedToUsers:
                  self.tapisSites[tapisSite]['state'] = 'restrictedByUser'

         groupMembership = GroupMembership(restrictionUser=restrictionUser)
         for tapisSite in self.tapisSites:
            if self.tapisSites[tapisSite]['state'] == 'enabled':
               restrictedToGroups = self.tapisSites[tapisSite]['restrictedToGroups']
               if len(restrictedToGroups) > 0:
                  groupOK = False
                  for restrictedToGroup in restrictedToGroups:
                     if groupMembership.isGroupMember(restrictedToGroup):
                        groupOK = True
                        break
                  if not groupOK:
                     self.tapisSites[tapisSite]['state'] = 'restrictedByGroup'


   def readTapis3SitesInfoFile(self,
                               tapisSitesInfoPath):
      tapisSitePattern = re.compile(r'(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern  = re.compile(r'( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern   = re.compile(r'\s*#.*')
      tapisSiteName    = ""

      if os.path.exists(tapisSitesInfoPath):
         try:
            fpInfo = open(tapisSitesInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   tapisSitePattern.match(record):
                        tapisSiteName = tapisSitePattern.match(record).group(2)
                        self.tapisSites[tapisSiteName] = {'venue':'',
                                                          'venuePort':22,
                                                          'venueMechanism':'',
                                                          'sshOptions':'',
                                                          'identityManagers':[],
                                                          'remoteUser':'',
                                                          'remoteBatchSystem':'TAPIS3',
                                                          'remoteBinDirectory':os.path.join('${HOME}','bin'),
                                                          'remoteApplicationRootDirectory':'',
                                                          'remoteScratchDirectory':'',
                                                          'siteMonitorDesignator':'',
                                                          'stageFiles':True,
                                                          'state':'enabled',
                                                          'restrictedToUsers':[],
                                                          'restrictedToGroups':[],
                                                          'appId':'',
                                                          'appVersion':'',
                                                          'queue':'',
                                                          'partition':'',
                                                          'stageSystem':'',
                                                          'archiveSystem':''
                                                         }
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.tapisSites[tapisSiteName]:
                           if   isinstance(self.tapisSites[tapisSiteName][key],list):
                              self.tapisSites[tapisSiteName][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.tapisSites[tapisSiteName][key],bool):
                              self.tapisSites[tapisSiteName][key] = bool(value.lower() == 'true')
                           elif isinstance(self.tapisSites[tapisSiteName][key],float):
                              self.tapisSites[tapisSiteName][key] = float(value)
                           elif isinstance(self.tapisSites[tapisSiteName][key],int):
                              self.tapisSites[tapisSiteName][key] = int(value)
                           elif isinstance(self.tapisSites[tapisSiteName][key],dict):
                              try:
                                 sampleKey   = list(self.tapisSites[tapisSiteName][key].keys())[0]
                                 sampleValue = self.tapisSites[tapisSiteName][key][sampleKey]
                              except:
                                 try:
                                    self.tapisSites[tapisSiteName][key] = json.loads(value)
                                 except:
                                    self.tapisSites[tapisSiteName][key] = {}
                                    sampleKey   = "key"
                                    sampleValue = "value"
                              else:
                                 self.tapisSites[tapisSiteName][key] = {}
               
                              if not self.tapisSites[tapisSiteName][key]:
                                 for e in value.split(','):
                                    dictKey,dictValue = e.split(':')
                                    if isinstance(sampleKey,int):
                                       dictKey = int(dictKey)
                                    if   isinstance(sampleValue,int):
                                       dictValue = int(dictValue)
                                    elif isinstance(sampleValue,float):
                                       dictValue = float(dictValue)
                                    elif isinstance(sampleValue,bool):
                                       dictValue = bool(dictValue.lower() == 'true')
                                    self.tapisSites[tapisSiteName][key][dictKey] = dictValue
                           else:
                              self.tapisSites[tapisSiteName][key] = value
                        else:
                           message = "Undefined key = value pair %s = %s for tapisSite %s" % (key,value,tapisSiteName)
                           self.logger.log(logging.WARNING,getLogMessage(message))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("Tapis3Sites configuration file %s could not be read" % \
                                                                                             (tapisSitesInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("Tapis3Sites configuration file %s could not be opened" % \
                                                                                            (tapisSitesInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("Tapis3Sites configuration file %s is missing" % \
                                                                                (tapisSitesInfoPath)))


   def getEnabledTapisSites(self):
      enabledTapisSites = []
      for tapisSite in self.tapisSites:
         if self.tapisSites[tapisSite]['state'] == 'enabled':
            enabledTapisSites.append(tapisSite)

      return(enabledTapisSites)


   def purgeDisabledTapisSites(self,
                               tapisSiteNames):
      reasonsDenied = {}
      markedForDeletion = []
      for tapisSiteName in tapisSiteNames:
         try:
            tapisSite = self.tapisSites[tapisSiteName]
            state     = tapisSite['state']
            if state != 'enabled':
               markedForDeletion.append(tapisSiteName)
               reasonsDenied[tapisSiteName] = state
         except:
            pass
      for tapisSiteName in markedForDeletion:
         tapisSiteNames.remove(tapisSiteName)
      del markedForDeletion

      return(reasonsDenied)


   def getSiteKeyValue(self,
                       tapisSiteName,
                       key):
      value = ""

      if tapisSiteName in self.tapisSites:
         tapisSite = self.tapisSites[tapisSiteName]
         if key in tapisSite:
            value = tapisSite[key]

      return(value)


   def getTapisSiteVenue(self,
                         tapisSiteName):
      venue = []
      if tapisSiteName in self.tapisSites:
         tapisSite = self.tapisSites[tapisSiteName]
         venue = tapisSite['venue']

      return(venue)


   def tapisSiteExists(self,
                       tapisSiteName):
      return(tapisSiteName in self.tapisSites)


   def getDefaultTapisSiteInfo(self):
      tapisSiteInfo = {}
      tapisSiteInfo['tapisName']                      = "default"
      tapisSiteInfo['venue']                          = "Unknown"
      tapisSiteInfo['venuePort']                      = 22
      tapisSiteInfo['venueMechanism']                 = ""
      tapisSiteInfo['sshOptions']                     = ""
      tapisSiteInfo['identityManagers']               = []
      tapisSiteInfo['remoteUser']                     = ""
      tapisSiteInfo['remoteBatchSystem']              = "TAPIS3"
      tapisSiteInfo['remoteBinDirectory']             = os.path.join('${HOME}','bin')
      tapisSiteInfo['remoteApplicationRootDirectory'] = ""
      tapisSiteInfo['remoteScratchDirectory']         = ""
      tapisSiteInfo['siteMonitorDesignator']          = ""
      tapisSiteInfo['sharedUserSpace']                = False
      tapisSiteInfo['stageFiles']                     = True
      tapisSiteInfo['appId']                          = ""
      tapisSiteInfo['appVersion']                     = ""
      tapisSiteInfo['queue']                          = ""
      tapisSiteInfo['partition']                      = ""
      tapisSiteInfo['stageSystem']                    = ""
      tapisSiteInfo['archiveSystem']                  = ""

      return(tapisSiteInfo)


   def getTapisSiteInfo(self,
                        tapisSiteName):
      tapisSiteInfo = {}

      try:
         tapisSite = self.tapisSites[tapisSiteName]
      except:
         tapisSite = self.getDefaultTapisSiteInfo()

      tapisSiteInfo['tapisName']                      = tapisSiteName
      tapisSiteInfo['venue']                          = tapisSite['venue']
      tapisSiteInfo['venuePort']                      = tapisSite['venuePort']
      tapisSiteInfo['venueMechanism']                 = tapisSite['venueMechanism']
      tapisSiteInfo['sshOptions']                     = tapisSite['sshOptions']

      tapisSiteInfo['identityManagers']               = tapisSite['identityManagers']
      tapisSiteInfo['remoteUser']                     = tapisSite['remoteUser']
      if   tapisSiteInfo['remoteUser'].startswith('USER:'):
         tapisSiteInfo['sharedUserSpace']             = True
      elif tapisSiteInfo['remoteUser'].startswith('USER'):
         tapisSiteInfo['sharedUserSpace']             = True
      else:
         tapisSiteInfo['sharedUserSpace']             = False

      tapisSiteInfo['remoteBatchSystem']              = tapisSite['remoteBatchSystem']
      tapisSiteInfo['remoteBinDirectory']             = tapisSite['remoteBinDirectory']
      tapisSiteInfo['remoteApplicationRootDirectory'] = tapisSite['remoteApplicationRootDirectory']
      tapisSiteInfo['remoteScratchDirectory']         = tapisSite['remoteScratchDirectory']
      tapisSiteInfo['siteMonitorDesignator']          = tapisSite['siteMonitorDesignator']
      tapisSiteInfo['stageFiles']                     = tapisSite['stageFiles']
      tapisSiteInfo['appId']                          = tapisSite['appId']
      tapisSiteInfo['appVersion']                     = tapisSite['appVersion']
      tapisSiteInfo['queue']                          = tapisSite['queue']
      tapisSiteInfo['partition']                      = tapisSite['partition']
      tapisSiteInfo['stageSystem']                    = tapisSite['stageSystem']
      if 'archiveSystem' in tapisSite:
         tapisSiteInfo['archiveSystem']               = tapisSite['archiveSystem']
      else:
         tapisSiteInfo['archiveSystem']               = ""

      return(tapisSiteInfo)


