# @package      hubzero-submit-distributor
# @file         AppsAccessInfo.py
# @copyright    Copyright (c) 2012-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2012-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
import os
import re
import glob
import json
import logging

from hubzero.submit.LogMessage      import getLogJobIdMessage as getLogMessage
from hubzero.submit.GroupMembership import GroupMembership

try:
   iterRange = xrange
except NameError as e:
   iterRange = range

class AppsAccessInfo:
   def __init__(self,
                appsAccessPath,
                restrictionUser=None):
      self.logger = logging.getLogger(__name__)

      self.appsAccess = {}
      self.whitelist  = {}
      self.blacklist  = {}

      environmentVariablePattern = re.compile(r'\$\{(.*?)\}')

      if os.path.isdir(appsAccessPath):
         for appsAccessInfoPath in glob.iglob(os.path.join(appsAccessPath,'*')):
            self.readAppsAccessInfoFile(appsAccessInfoPath)
      else:
         for appsAccessInfoPath in glob.iglob(appsAccessPath):
            self.readAppsAccessInfoFile(appsAccessInfoPath)

      groupMembership = GroupMembership(restrictionUser=restrictionUser)
      markedForDeletion = []
      for groupName in self.appsAccess:
         if not groupMembership.isGroupMember(groupName):
            markedForDeletion.append(groupName)
      for groupName in markedForDeletion:
         del self.appsAccess[groupName]
      del markedForDeletion

      for groupName in self.appsAccess:
         markedForDeletion = []
         for groupIndex in iterRange(len(self.appsAccess[groupName])):
            state          = self.appsAccess[groupName][groupIndex]['state']
            classification = self.appsAccess[groupName][groupIndex]['classification']
            if state == 'disabled' or classification == '':
               markedForDeletion.append(groupIndex)
         markedForDeletion.reverse()
         for groupIndex in markedForDeletion:
            del self.appsAccess[groupName][groupIndex]
         del markedForDeletion

      for groupName in self.appsAccess:
         for groupIndex in iterRange(len(self.appsAccess[groupName])):
            for element in self.appsAccess[groupName][groupIndex]['whitelist']:
               element = os.path.expandvars(os.path.expanduser(element))
               if not re.search(environmentVariablePattern,element):
                  priority       = self.appsAccess[groupName][groupIndex]['priority']
                  classification = self.appsAccess[groupName][groupIndex]['classification']
                  if not priority in self.whitelist:
                     self.whitelist[priority] = {}
                  if not element in self.whitelist[priority]:
                     self.whitelist[priority][element] = classification

            for element in self.appsAccess[groupName][groupIndex]['blacklist']:
               element = os.path.expandvars(os.path.expanduser(element))
               if not re.search(environmentVariablePattern,element):
                  priority       = self.appsAccess[groupName][groupIndex]['priority']
                  classification = self.appsAccess[groupName][groupIndex]['classification']
                  if not priority in self.blacklist:
                     self.blacklist[priority] = {}
                  if not element in self.blacklist[priority]:
                     self.blacklist[priority][element] = classification


   def readAppsAccessInfoFile(self,
                              appsAccessInfoPath):
      groupPattern    = re.compile(r'(\s*\[)([^\s]*)(]\s*)')
      keyValuePattern = re.compile(r'( *)(\w*)( *= *)(.*[^\s$])( *)')
      commentPattern  = re.compile(r'\s*#.*')
      groupName       = ""

      if os.path.exists(appsAccessInfoPath):
         try:
            fpInfo = open(appsAccessInfoPath,'r')
            try:
               eof = False
               while not eof:
                  record = fpInfo.readline()
                  if record != "":
                     record = commentPattern.sub("",record)
                     if   groupPattern.match(record):
                        groupName = groupPattern.match(record).group(2)
                        if not groupName in self.appsAccess:
                           self.appsAccess[groupName] = []
                        groupIndex = len(self.appsAccess[groupName])
                        self.appsAccess[groupName].append({'whitelist':[],
                                                           'blacklist':[],
                                                           'priority':0,
                                                           'classification':'',
                                                           'state':'enabled'
                                                          })
                     elif keyValuePattern.match(record):
                        key,value = keyValuePattern.match(record).group(2,4)
                        if key in self.appsAccess[groupName][groupIndex]:
                           if   isinstance(self.appsAccess[groupName][groupIndex][key],list):
                              self.appsAccess[groupName][groupIndex][key] = [e.strip() for e in value.split(',')]
                           elif isinstance(self.appsAccess[groupName][groupIndex][key],bool):
                              self.appsAccess[groupName][groupIndex][key] = bool(value.lower() == 'true')
                           elif isinstance(self.appsAccess[groupName][groupIndex][key],float):
                              self.appsAccess[groupName][groupIndex][key] = float(value)
                           elif isinstance(self.appsAccess[groupName][groupIndex][key],int):
                              self.appsAccess[groupName][groupIndex][key] = int(value)
                           elif isinstance(self.appsAccess[groupName][groupIndex][key],dict):
                              try:
                                 sampleKey   = list(self.appsAccess[groupName][groupIndex][key].keys())[0]
                                 sampleValue = self.appsAccess[groupName][groupIndex][key][sampleKey]
                              except:
                                 try:
                                    self.appsAccess[groupName][groupIndex][key] = json.loads(value)
                                 except:
                                    self.appsAccess[groupName][groupIndex][key] = {}
                                    sampleKey   = "key"
                                    sampleValue = "value"
                              else:
                                 self.appsAccess[groupName][groupIndex][key] = {}
               
                              if not self.appsAccess[groupName][groupIndex][key]:
                                 for e in value.split(','):
                                    dictKey,dictValue = e.split(':')
                                    if isinstance(sampleKey,int):
                                       dictKey = int(dictKey)
                                    if   isinstance(sampleValue,int):
                                       dictValue = int(dictValue)
                                    elif isinstance(sampleValue,float):
                                       dictValue = float(dictValue)
                                    elif isinstance(sampleValue,bool):
                                       dictValue = bool(dictValue.lower() == 'true')
                                    self.appsAccess[groupName][groupIndex][key][dictKey] = dictValue
                           else:
                              self.appsAccess[groupName][groupIndex][key] = value
                        else:
                           self.logger.log(logging.WARNING,getLogMessage("Undefined key = value pair %s = %s for group %s" % \
                                                                                                     (key,value,groupName)))
                  else:
                     eof = True
            except (IOError,OSError):
               self.logger.log(logging.ERROR,getLogMessage("AppsAccess configuration file %s could not be read" % \
                                                                                             (appsAccessInfoPath)))
            finally:
               fpInfo.close()
         except (IOError,OSError):
            self.logger.log(logging.ERROR,getLogMessage("AppsAccess configuration file %s could not be opened" % \
                                                                                            (appsAccessInfoPath)))
      else:
         self.logger.log(logging.ERROR,getLogMessage("AppsAccess configuration file %s is missing" % \
                                                                                (appsAccessInfoPath)))


   def isSubmissionAllowed(self,
                           executablePath):
      submissionAllowed = False
      classification = None
      expandedPath = os.path.expandvars(os.path.expanduser(executablePath))

      try:
         canonicalPath = os.path.realpath(expandedPath)
      except (IOError,OSError):
         self.logger.log(logging.ERROR,getLogMessage("isSubmissionAllowed:os.getcwd(): No such file or directory"))
      else:
         if os.path.isfile(canonicalPath):
            if os.access(canonicalPath,os.X_OK) and os.access(canonicalPath,os.R_OK):
               submissionAllowed = True

      if submissionAllowed:
         blacklistPriority = -1
         priorities = list(self.blacklist.keys())
         priorities.sort(reverse=True)
         for priority in priorities:
            for element in self.blacklist[priority]:
               if re.match(element,expandedPath) or re.match(element,canonicalPath):
                  blacklistPriority       = priority
                  blacklistClassification = self.blacklist[priority][element]
                  break
            if blacklistPriority > 0:
               break

         whitelistPriority = -1
         priorities = list(self.whitelist.keys())
         priorities.sort(reverse=True)
         for priority in priorities:
            for element in self.whitelist[priority]:
               if re.match(element,expandedPath) or re.match(element,canonicalPath):
                  whitelistPriority = priority
                  whitelistClassification = self.whitelist[priority][element]
                  break
            if whitelistPriority > 0:
               break

         if blacklistPriority >= whitelistPriority:
            submissionAllowed = False
         else:
            classification = whitelistClassification

      return(submissionAllowed,classification)


