#!/usr/bin/env python2
#
# @package      hubzero-submit-monitors
# @file         monitorCloud.py
# @copyright    Copyright (c) 2004-2020 The Regents of the University of California.
# @license      http://opensource.org/licenses/MIT MIT
#
# Copyright (c) 2004-2020 The Regents of the University of California.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# HUBzero is a registered trademark of The Regents of the University of California.
#
# ----------------------------------------------------------------------
#  monitorCloud.py
#
#  script which monitors a cloud instance for use with distributor and related processes.
#
import sys
import os
import select
import signal
import socket
import time
import logging
from errno import EINTR

from hubzero.submit.LogMessage   import getLogPIDMessage as getLogMessage
from hubzero.submit.DaemonsInfo  import DaemonsInfo
from hubzero.submit.InfosInfo    import InfosInfo
from hubzero.submit.CloudMonitor import CloudMonitor

MONITORROOT = os.path.join(os.sep,'opt','submit')

MONITORLOGLOCATION = os.path.join(os.sep,'var','log','submit','monitors')
MONITORLOGFILENAME = "monitorCloud.log"
APPLICATIONLOGGER  = logging.getLogger('')

PIDFILE = os.path.join(os.sep,'var','run','submit','submit-cloud.pid')

DUMPFILENAME = "monitorCloud.dump"
DUMPFILEPATH = os.path.join(MONITORROOT,DUMPFILENAME)

CONFIGURATIONDIRECTORY   = os.path.join(os.sep,'etc','submit')
DAEMONSCONFIGURATIONFILE = 'daemons.conf'
INFOSCONFIGURATIONFILE   = 'infos.conf'

ACTIVITYUPDATEINTERVAL = 15.*60.

def openLogger(logDirectory,
               hubLogFile):
   class EmptyFilter(logging.Filter):
      """
      This is a filter which rejects empty messages

      """

      def filter(self,record):
         if record.getMessage() == "":
            emptyRecord = True
         else:
            emptyRecord = False

         return(not emptyRecord)

   APPLICATIONLOGGER.setLevel(logging.DEBUG)

   hubLogPath = os.path.join(logDirectory,hubLogFile)
   logHandler = logging.FileHandler(hubLogPath)
   fdLogFile = logHandler.stream.fileno()

   emptyFilter = EmptyFilter()
   logHandler.addFilter(emptyFilter)

   logFormatter = logging.Formatter('%(asctime)s %(message)s','%s [%a %b %d %H:%M:%S %Y]')
   logHandler.setFormatter(logFormatter)
   APPLICATIONLOGGER.addHandler(logHandler)

   return(fdLogFile)


def daemonize(fdLogFile):
   if fdLogFile != sys.stdout.fileno():
      try:
         devnull = open("/dev/null",'r')
         try:
            os.dup2(devnull.fileno(),sys.stdin.fileno())
            os.dup2(fdLogFile,sys.stdout.fileno())
            os.dup2(fdLogFile,sys.stderr.fileno())
         except OSError:
            APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("file descriptor dup failed"))
      except (IOError,OSError):
         APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("%s could not be opened" % ("/dev/null")))

   if os.fork() != 0:
      os.wait()
      sys.exit(0)
   else:
      os.setsid()
      os.chdir("/")
      pid = os.fork()
      if pid != 0:
         sys.exit(0)

   time.sleep(2)


def writePidFile(pidFile):
   pidSaved = False
   try:
      fpPid = open(pidFile,'w')
      try:
         fpPid.write("%d\n" % (os.getpid()))
      except (IOError,OSError):
         APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Unable to write pid (%d) to %s" % (os.getpid(),pidFile)))
      else:
         pidSaved = True
      finally:
         fpPid.close()
   except:
      APPLICATIONLOGGER.log(logging.ERROR,getLogMessage("Unable to open pid (%d) to %s" % (os.getpid(),pidFile)))

   if not pidSaved:
      os._exit(os.EX_CANTCREAT)


class MonitorCloud:
   def __init__(self,
                installedDirectory,
                configurationDirectory,
                daemonsConfigurationFile,
                infosConfigurationFile,
                dumpFilePath):
      self.logger = logging.getLogger(__name__)

      self.binDirectory  = os.path.join(installedDirectory,'bin')
      os.environ['PATH'] = self.binDirectory + os.pathsep + os.environ['PATH']

      configFilePath = os.path.join(configurationDirectory,daemonsConfigurationFile)
      daemonsInfo    = DaemonsInfo(configFilePath)
      self.listenURI = daemonsInfo.getDaemonListenURI('cloudMonitor','tcp')

      self.dumpFilePath = dumpFilePath

      configFilePath = os.path.join(configurationDirectory,infosConfigurationFile)
      self.infosInfo = InfosInfo(configFilePath)
      self.cloudMonitor = CloudMonitor(self.listenURI,
                                       self.infosInfo.getInfoPath('clouds'),
                                       activeCloudDumpPath=self.dumpFilePath)

      self.terminateCloudMonitor = False

      self.exitSystemCallOnInterrupt = False

      signal.signal(signal.SIGINT,self.sigINT_handler)
      signal.signal(signal.SIGHUP,self.sigHUP_handler)
      signal.signal(signal.SIGQUIT,self.sigQUIT_handler)
      signal.signal(signal.SIGABRT,self.sigABRT_handler)
      signal.signal(signal.SIGTERM,self.sigTERM_handler)


   def terminate(self):
      self.terminateCloudMonitor = True


   def sigGEN_handler(self,
                      signalNumber,
                      frame):
      self.terminate()
      if self.exitSystemCallOnInterrupt:
         if sys.version_info > (3,):
            raise InterruptedError(EINTR,"System Call Interrupt")


   def sigINT_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGINT!"))
      self.cloudMonitor.reloadCloudsInfo()


   def sigHUP_handler(self,
                      signalNumber,
                      frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGHUP!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigQUIT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGQUIT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigABRT_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGABRT!"))
      self.sigGEN_handler(signalNumber,frame)


   def sigTERM_handler(self,
                       signalNumber,
                       frame):
      self.logger.log(logging.INFO,getLogMessage("Received SIGTERM!"))
      self.sigGEN_handler(signalNumber,frame)


   def monitor(self):
      if not self.cloudMonitor.isListening():
         self.logger.log(logging.ERROR,getLogMessage("Port binding failed"))
         sys.exit(1)

      self.logger.log(logging.INFO,getLogMessage("********************************"))
      self.logger.log(logging.INFO,getLogMessage("* cloud server monitor started *"))
      self.logger.log(logging.INFO,getLogMessage("********************************"))

      while True:
         listeningSocket,activeReaders = self.cloudMonitor.getInputObjects()
         activeWriters                 = self.cloudMonitor.getOutputObjects()
         if not listeningSocket and not activeReaders and not activeWriters:
            self.logger.log(logging.INFO,getLogMessage("********************************"))
            self.logger.log(logging.INFO,getLogMessage("* cloud server monitor stopped *"))
            self.logger.log(logging.INFO,getLogMessage("********************************"))
            break

         try:
            self.exitSystemCallOnInterrupt = True
            readyReaders,readyWriters,readyExceptions = select.select(listeningSocket+activeReaders,
                                                                      activeWriters,
                                                                      [],
                                                                      ACTIVITYUPDATEINTERVAL)
         except select.error as e:
            if e.args[0] == EINTR:
               readyReaders = []
               readyWriters = []
            else:
               self.terminate()
         finally:
            self.exitSystemCallOnInterrupt = False

         for readyReader in readyReaders:
            if   readyReader in listeningSocket:
               if not self.cloudMonitor.acceptConnection():
                  self.logger.log(logging.ERROR,getLogMessage("Connection failed."))
            elif readyReader in activeReaders:
               self.cloudMonitor.receiveMessage(readyReader)

         self.cloudMonitor.processRequests()

         for readyWriter in readyWriters:
            if readyWriter in activeWriters:
               self.cloudMonitor.sendMessage(readyWriter)

         self.cloudMonitor.purgeIdleInstances()

         if self.terminateCloudMonitor:
            self.cloudMonitor.terminate()


if __name__ == '__main__':

   fdLogFile = openLogger(MONITORLOGLOCATION,MONITORLOGFILENAME)
   daemonize(fdLogFile)

   writePidFile(PIDFILE)

   __monitorCloud__ = MonitorCloud(MONITORROOT,CONFIGURATIONDIRECTORY,
                                               DAEMONSCONFIGURATIONFILE,INFOSCONFIGURATIONFILE,
                                               DUMPFILEPATH)
   __monitorCloud__.monitor()


